/** $Header: cdi_stability.c,v 1.21 99/05/21 13:30:50 jaaps Exp $
********************************** - - ****************************************
* 						   
*         _/_/_/_/_/_/    _/_/_/    _/_/      _/_/       SRON Utrecht
*      _/_/_/_/  _/_/  _/_/_/_/_/  _/_/_/  _/_/_/    
*     _/_/_/        _/_/      _/_/  _/_/_/_/_/       _/     _/ _/_/_/_/ _/_/_/
*      _/_/_/_/_/  _/_/_/_/_/_/_/    _/_/_/         _/     _/ _/     _/_/
*           _/_/_/_/_/_/_/_/_/_/  _/_/_/_/_/  _/_/ _/  _/ _/ _/_/_/ _/      
*  _/_/  _/_/_/_/_/_/      _/_/_/_/_/  _/_/_/     _/_/ _/_/ _/     _/_/  
* _/_/_/_/_/_/  _/_/      _/_/_/_/      _/_/     _/     _/ _/       _/_/_/   
*
*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -
* Use	      :	cdi_stability ();
*
* Parameters  :	
*
*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -
* Dependencies:	
*
*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -
* Description :	scan attitude data over the selected period,
*		create a 2D pointing histogram
*
*  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -  -
* $Log:	cdi_stability.c,v $
 * Revision 1.21  99/05/21  13:30:50  13:30:50  jaaps (Jaap Schuurmans)
 *   cdi_stability--att_open: do not try to open files named "-".
 * 
 * Revision 1.21  99/05/21  12:28:03  12:28:03  jaaps (Jaap Schuurmans)
 *    cdi_stability--att_open: do not try to open files named "-".
 * 
 * Revision 1.20  98/07/01  14:10:05  14:10:05  jaaps (Jaap Schuurmans)
 * non essential, cosmetic adjustments (no reason found in source).
 * 
* Revision 1.19  1998/06/18 07:42:17  jaaps
* non essential, cosmetic adjustments (no reason found in source).
*
* Revision 1.18  1998/05/13 10:28:54  jaaps
*    att_open: gracefully handle 0 length attitude datafiles, did cause abort.
*
* Revision 1.17  1997/09/10 14:56:01  jaaps
* non essential, cosmetic adjustments (no reason found in source).
*
* Revision 1.15  1997/09/10 10:34:26  jaaps
* no entry
*
* Revision 1.13  1997/09/10 07:59:10  jaaps
* non essential, cosmetic adjustments (no reason found in source).
*
* Revision 1.12  1997/09/09 13:59:41  jaaps
*    cdi_finistab: as before, there is no reason not to treat the northangle
*    		 likewise, ecco.
*
* Revision 1.11  1997/09/09 13:52:36  jaaps
*   cdi_finistab: apply any sax/wfc misalignment correction not only to x/y mean
*   		but to found range also. (mean sometimes shifts outside range).
*
* Revision 1.9  1997/09/08 08:25:43  jaaps
*   cdi_initstab: preserve the initial pointing, used as reference in attitude drift
*   		computation.
*
* Revision 1.8  1997/06/18 14:05:49  jaaps
*    cdi_finistab: apply the dx, dy camera misalignment corrections.
*
* Revision 1.7  1997/06/04 15:21:50  jaaps
* non essential, cosmetic adjustments (no reason found in source).
*
* Revision 1.6  1997/04/18 14:22:08  jaaps
*   cdi_stability: all attitude data were inadvertendly skipped if the attitude time span
*   		 was fully within 1 single event timewindow, fixed.
*   att_open     : failure may also arise from 'no events in timewindow' on a file that
* 		 could be opened, error message changed to report this possibility.
*
* Revision 1.5  1997/04/11 15:22:30  jaaps
*   cdi_cna      : repair omission in na computation (add 90 deg).
*   cdi_stability: give correct units in info output.
*
* Revision 1.4  1997/03/13 09:44:51  jaaps
*    cdi_stability: changed pointing-stability data unit from 'detector-pixel'
* 		  to millimeter.
*
* Revision 1.3  1997/03/10 08:58:58  jaaps
* RCS-test
*
* Revision 1.2  1997/03/10 08:49:35  jaaps
* non essential, cosmetic adjustments (no reason found in source).
*
* Revision 1.1  1997/03/10 08:36:57  jaaps
* Initial revision
*
********************************** - - ****************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <math.h>
#include <time.h>

#include "wfcglobals.h"
#include "fits.h"
#include "util.h"

#include "instrument.h"

#include "expose.h"

#define C_CALLOC(s,a,b) secure_alloc (#s, a, b)


static char *RCSID = "$Header: cdi_stability.c,v 1.21 99/05/21 13:30:50 jaaps Exp $";

static char *cdi_stab_dbgcv (char *s, double t) {
   
   double	msec;
   int		hh, mm, ss;

   *s= '\0';  
   if (t > 0.0) {
      msec= 24.0*fmod (t, 1.0);
      hh= (int) (msec); msec=   60.0*(msec - (double) hh);
      mm= (int) (msec); msec=   60.0*(msec - (double) mm);
      ss= (int) (msec); msec= 1000.0*(msec - (double) ss)+0.5;
      if ((int) msec > 999) {
	 msec= 0.0; ss += 1;
	 if (ss == 60) {
	    ss= 0; mm += 1; 
	    if (mm == 60) { mm= 0; hh = (hh+1)%24;}
	 }
      }
      sprintf (s, " (%2.2d:%2.2d:%2.2d.%3.3d)", hh, mm, ss, (int) msec);
   }
   return s;
 }

static void cdi_dbgout_hdr (cardinal Lvl) {
   
   wLog (Lvl, "    t rel        mjd          (h:m:s)      object      event       exposure-time\n");
}

static void cdi_stab_dbgout (double t, double start, char *name, char *action, double est) {
   
   char		s0[256], s1[64], s2[64], s3[64];
   double	msec;
   int		hh, mm, ss;
   
   strcpy (s1, "    ----        ----    "); 
   if (t != 0.0 || start != 0.0) sprintf (s1, "+%10.3f %12.6f", (t-start)*WFC_SECpDAY, t);
   cdi_stab_dbgcv (s2, t);
   
   *s3= '\0'; if (est > 0.0) sprintf (s3, " et += %9.3f sec.", (t-est)*WFC_SECpDAY);
   sprintf (s0, "%24.24s %12s %-10.10s %10.10s %s\n", 
	    s1,
	    s2,
	    (name   != NULL) ?   name : " ",
	    (action != NULL) ? action : " ",
	    s3);
   wLog (DBG_CONT, s0);
}
			   
static char *cdi_attfromevl (char *r, WFC_EXPOSEPTR exp, cardinal len) {
   
   char *n, *s;
   char template[]= "W%s_%4.4dw%1.1datt1.fits";
   
   *r= '\0';
   if (exp != NULL) {
      strncpy (r, exp->evtlist[0], len);
      if ((n= strrchr (r, '/')) == NULL) n= r; if (*n == '/') n++;
      
      sprintf (n, template, exp->obsPeriod, 0, exp->wfc);
   }
   /***
   ****   if  ((s= strstr (n, "prn")) == NULL) s= strstr (n, "srn");
   ****  if (s != NULL && s[3] >= '0' && s[3] <= '9') strncpy (s, "att1", 4);
   ***/
   return r;
}

WFC_EVENTFILEPTR att_open (char *name, WFC_EXPOSEPTR exp, WFC_FILTERPTR	fltr) {
   
   WFC_EVENTFILEPTR	afp= NULL;
   char			mess[256], afname[WFCMINPATHLENGTH], ts[256];
   
   if (name == NULL || *name == '\0') 
      name= cdi_attfromevl (afname, exp, WFCMINPATHLENGTH);
   
   if (strcmp (name, "-") == 0) return NULL;
   afp= evt_init (name,
		  fltr,
		  NULL /** WFC_XCBFNCPTR ocb  **/,
		  NULL /** WFC_XCBFNCPTR ccb  **/,
		  NULL /** void   *argcb ccb  **/,
		  0    /** cardinal  bufSize  **/);
   if (afp == NULL || afp->f == NULL) {
      sprintf (mess,
	       "att_open: \"%s\" attitude, cannot open or no events in timewindow.\n", name);
      wLog (DBG_ERROR, mess);
      sprintf (mess,
	       "          use \"-b attfile\" (parameter attfile) to override default.\n");
      wLog (DBG_CONT, mess);
      sprintf (mess,
	       "          exact source exposure time not established.\n");
      wLog (DBG_CONT, mess);
      return NULL;
   }
   if (exp != NULL && exp->wfc != 0 &&  afp->wfc != exp->wfc) {
      sprintf (mess,  
	       "att_open: \"%s\" attitude data for wfc%d, event data from wfc%d.\n",
	       afp->evt->evtfile, afp->wfc, exp->wfc);
      wLog (DBG_ERROR, mess);
      wLog (DBG_CONT, "********! MIXING DATA FROM BOTH Wide Field Camera's !********\n\n");
   }   
   if (exp != NULL && exp->obsPeriod != NULL && afp->obsPeriod != NULL && 
       strcmp (exp->obsPeriod, afp->obsPeriod) != 0) {
      sprintf (mess,  
	       "att_open: attitude data from obs.period %s (\"%s\"),\n",
	       afp->obsPeriod, afp->namelist[0]);
      wLog (DBG_ERROR, mess);
      sprintf (mess,  
	       "          event data from obs.period %s (\"%s\").\n",
	       exp->obsPeriod, exp->expfile);
      wLog (DBG_CONT, mess);
      wLog (DBG_CONT, "********! MIXING DATA FROM DIFFERENT OBSERVATION PERIODS !********\n\n");
   }
   if (fltr		!= NULL &&
       fltr->t		!= NULL && 
       afp->evt		!= NULL	&& 
       (afp->evt->mjds - fltr->t->r.s) > (1.0/WFC_SECpDAY) ) {
      wLog (DBG_ERROR, "att_open: possibly event data before start of attitude!!\n");
      sprintf (mess,   "          time window starts at %.6f\n", fltr->t->r.s);
      wLog (DBG_CONT, mess);
      sprintf (mess,   "          attitude data starts at %.6f, ie. %.3f sec. later\n",
	       afp->evt->mjds, (afp->evt->mjds - fltr->t->r.s)*WFC_SECpDAY);
      wLog (DBG_CONT, mess);
   }
   
   return afp;
}

double cdi_cna (double ra1, double dec1, double ra2, double dec2) {
   
   double d1, d2, a, b, aa, sd, cd, bigk, smlk, dk, na;
      
      
   aa = WFC_DEG2RAD(ra1-ra2); 
   d1 = WFC_DEG2RAD(dec1);
   d2 = WFC_DEG2RAD(dec2);
   
   cd= cos (d2)     ; sd  = sin (d2); 
   a = cd*cos (aa)  ; bigk= atan2 (a, sd); smlk= sqrt(a*a+sd*sd);
   a = cd*sin (aa)  ; dk  = d1+bigk;       b   = smlk * cos (dk);
   
   na = WFC_RAD2DEG(atan2 (a, b)) + 90.0;
   
   na= fmod (na, 360.0);
   return (na < 0.0) ? na+360.0 : na;	  
}

void cdi_addstab (WFC_STABILTYPTR sb, ATT_CAMERAPTR att, WFC_INSTRUMENTPTR ins, cardinal wfc) {
   
   float	x, y, h;
   double 	raZ, raY, decZ, decY, na;
   
   
   raZ = att->ra_scy ; if (wfc == 1) raZ += 180.0; if (raZ >= 360.0) raZ -= 360.0;
   decZ= att->dec_scy; if (wfc == 1) decZ = -decZ;
   raY = att->ra_scx+180.0;	 		   if (raY >= 360.0) raY -= 360.0;
   decY= -att->dec_scx;
   na  = cdi_cna (raZ, decZ, raY, decY); 
   
   if (sb->n == 0) {
      sb->ra = raZ;
      sb->dec= decZ; 
      sb->na = na;
      set_trafomatrix_ceim (&sb->ra, &sb->dec, &sb->na);
   }
   
   trafo_ceim (&raZ, &decZ, &x, &y, 1000.0);
   x=  sb->efrq*x*ins->focallength/ins->sim.esize.x; 
   y=  sb->efrq*y*ins->focallength/ins->sim.esize.y;
      
   if (sb->n == 0) {
      sb->xrng.L = sb->xrng.H = x;
      sb->yrng.L = sb->yrng.H = y;
      sb->narng.L= sb->narng.H= na;
   }
   if (x  < sb->xrng.L)  sb->xrng.L = x;  if (x  > sb->xrng.H)  sb->xrng.H = x;
   if (y  < sb->yrng.L)  sb->yrng.L = y;  if (y  > sb->yrng.H)  sb->yrng.H = y;
   if (na < sb->narng.L) sb->narng.L= na; if (na > sb->narng.H) sb->narng.H= na;
   
   sb->sx += x ;  sb->ssx += x*x;
   sb->sy += y ;  sb->ssy += y*y;
   sb->sna+= na;  sb->ssna+= na*na;
   sb->n++;
   
   h= 0.5*sb->ms;
   if (y >= -h && y < h && x >= -h && x < h) {
      *dta_pixel (sb->map, (cardinal) (x+h), (cardinal) (y+h), 0, 0) += 1.0;
   }
   else {
      if (x < -h || x >= h) sb->overflow.x += 1.0;
      if (y < -h || y >= h) sb->overflow.y += 1.0;
   }
}

WFC_STABILTYPTR cdi_initstab (WFC_EXPOSEPTR 	exp,
			      WFC_INSTRUMENTPTR ins,
			      cardinal 		n,
			      float 		res)   /** # of mappixels to fit in 1 detector pixel. **/
{
   WFC_STABILTYPTR sb;
   cardinal	   c, len[2];
   float	   f;

   /**
   *** sb->ms: number of pixels (nx as well as ny) in map.
   ***         make sure sb->ms has 2 odd factors, facilitating a binoperation
   ***	       to detector pixel size.
   **/
   
   sb= C_CALLOC (cdi_newstab, 1, sizeof (WFC_STABILTYREC));


   if (res > 1.0) 	res= 1.0;
   if (res < 1.0/(float) (MAXTABMAPSIZE)) res= 1.0/(float) (MAXTABMAPSIZE);
   
   sb->ms   = ((cardinal) (1.0/res))|1;  /** # mappixels for 1 detector pixel, MUST be odd **/
   sb->efrq = sb->ms;
   
   sb->ares = 60.0*WFC_RAD2DEG(atan (ins->sim.esize.x/sb->efrq/ins->focallength));
   
   c= (cardinal) (MAXTABMAPSIZE/sb->ms);
   if (c > 0 && (c&1) == 0) c -= 1;
   sb->ms = c*sb->ms;
  
   sb->ra =  sb->ra_sat= exp->ra;
   sb->dec= sb->dec_sat= exp->dec; 
   sb->na =  sb->na_sat= exp->na;

   sb->n = 0;  
   sb->sx= sb->ssx= sb->sy= sb->ssy= sb->sna= sb->ssna= 0.0;
   len[0]= 
      len[1]= sb->ms; 
   sb->map  = dta_create ("stable", BXREAL, TRUE, 2, len);

   return sb;
}			

void cdi_finistab (WFC_STABILTYPTR sb, WFC_INSTRUMENTPTR ins) {
   
   float	nr, nf, x, y, ax, ay, ro;
   char		mess[256];
   rpair	d;

   sb->mean.x= sb->mean.y= sb->namean= 0.0;
   
   if (sb->n > 0) {
      
      ax= ins->sim.esize.x/sb->efrq; ay= ins->sim.esize.y/sb->efrq;
      
      sb->xrng.L *= ax; sb->xrng.H *= ax;
      sb->yrng.L *= ay; sb->yrng.H *= ay;
      
      nr= (float) sb->n;
      sb->mean.x= ax*sb->sx/nr;  sb->mean.y= ay*sb->sy/nr;
      sb->namean= sb->sna/nr;
      sb->psubj = "BeppoSAX Y-axis";
      
      if (sb->n > 1) {
	 sb->sigma.x= ax*sqrt ((sb->ssx - sb->sx*sb->sx/nr)/(nr-1.0));
	 sb->sigma.y= ay*sqrt ((sb->ssy - sb->sy*sb->sy/nr)/(nr-1.0));
	 sb->nasigma= sqrt ((sb->ssna - sb->sna*sb->sna/nr)/(nr-1.0));
      }
      dta_vs (OMUL, sb->map, 1.0/nr);			/** normalise **/
      sb->overflow.x /= nr;    sb->overflow.y /= nr;
      
      sb->ra = sb->ra_sat; sb->dec= sb->dec_sat; sb->na = sb->na_sat;
      sb->attcn.x= sb->attcn.y= 0.0;
      
      if (sb->overflow.x > 0 || sb->overflow.y > 0) {
	 sprintf (mess, 
		  "cdi_finistab: attitude map overflow, (x %.2f, y %.2f) %% of total exposure\n",
		  sb->overflow.x*100.0, sb->overflow.y*100.0);
	 wLog (DBG_WARNING, mess);
      }

   }
}

static void cdi_align (WFC_STABILTYPTR sb, WFC_INSTRUMENTPTR ins) {
   
   float	nr, nf, x, y, ax, ay, ro;
   char		mess[256];
   rpair	d;
   
   d.x= sb->mean.x;   d.y= sb->mean.y;
   ro = 0.0;
   
   sb->attcn.x= sb->attcn.y= sb->attcnna= 0.0;   
   if (ins->sim.det->align != NULL && ins->sim.det->align->align != NULL) {
      sb->psubj  = "WFC Y-axis";
      sb->attcn.x= ins->sim.det->align->align->dx;
      sb->attcn.y= ins->sim.det->align->align->dy;
      sb->attcnna= ins->sim.det->align->align->roll;
   }
   
   d.x= (sb->mean.x-sb->attcn.x)/ins->focallength;
   d.y= (sb->mean.y-sb->attcn.y)/ins->focallength;
   ro = sb->attcnna;
   jz_shiftPointing (d, ro, &sb->ra, &sb->dec, &sb->na);
   
   sprintf (mess,
	    "cdi_alignment: used wfc alignment correction [%.3f, %.3f, %.3f].\n",
	    sb->attcn.x, sb->attcn.y, sb->attcnna);
   wLog (DBG_L5, mess);
   
}
   
   
void cdi_alignreport (WFC_STABILTYPTR sb, WFC_INSTRUMENTPTR ins) {
   char		mess[256];
   time_t	st;
   
   wLog (DBG_CHAT, "================ ~~~ ================\n");
   st = (time_t) ((sb->mjds - asla_cldj ("1/1/70", "0:0:0"))*WFC_SECpDAY);
   sprintf (mess, "cdi_stability: start (UT) %s", asctime (gmtime (&st)));
   wLog (DBG_L2, mess);
   
   st = (time_t) ((sb->mjde - asla_cldj ("1/1/70", "0:0:0"))*WFC_SECpDAY);
   sprintf (mess, "                 end (UT) %s", asctime (gmtime (&st)));
   wLog (DBG_CONT, mess);
   
   sprintf (mess, "         statistics  (%d attitude records processed).\n", sb->n);
   wLog (DBG_CONT, mess);
   
   wLog (DBG_CONT, "                x                  y                   na\n");
   sprintf (mess, "  mean:      %6.2f (mm.)       %6.2f (mm.)        %6.2f (deg)\n",
	    sb->mean.x, sb->mean.y, sb->namean);
   wLog (DBG_CONT, mess);
   sprintf (mess, "st.dev:      %6.2f (mm.)       %6.2f (mm.)        %6.2f (min)\n",
	    sb->sigma.x, sb->sigma.y, 60.0*sb->nasigma);
   wLog (DBG_CONT, mess);
   sprintf (mess, "   min:      %6.2f (mm.)       %6.2f (mm.)        %6.2f (deg)\n",
	    sb->xrng.L, sb->yrng.L, sb->narng.L);
   wLog (DBG_CONT, mess);
   sprintf (mess, "   max:      %6.2f (mm.)       %6.2f (mm.)        %6.2f (deg)\n",
	    sb->yrng.H, sb->yrng.H, sb->narng.H);
   wLog (DBG_CONT, mess);
   sprintf (mess, " align:      %6.2f (mm.)       %6.2f (mm.)        %6.2f (deg)\n",
	    sb->attcn.x, sb->attcn.y, sb->attcnna);
   wLog (DBG_CONT, mess);
   wLog (DBG_CONT, "     -----------     ----------     ----------\n");
   sprintf (mess, "   sat: ra %.4f   dec %.4f   na %.4f\n",
	    sb->ra_sat, sb->dec_sat, sb->na_sat);
   wLog (DBG_CONT, mess);
   sprintf (mess, "   wfc: ra %.4f   dec %.4f   na %.4f\n",
	    sb->ra, sb->dec, sb->na);
   wLog (DBG_CONT, mess);
   wLog (DBG_CONT, "================ ~~~ ================\n");
}

void cdi_stability (char 		*name, 
		    WFC_INSTRUMENTPTR 	ins,
		    WFC_EXPOSEPTR	exp,
		    WFC_FILTERPTR	fltr,
		    float		rsln)
{
   
   ATT_CAMERAPTR	att;
   WFC_EVENTFILEPTR	afp;
   char			ts[256];
   bool			Trace= FALSE;
   cardinal		oowc= 0, oowtc= 0;
   char			mess[256];
   
   exp->sbp= NULL;
   if (ins == NULL) {
      wLog (DBG_L2, "cdi_stability: no instrument data, pointing drift not determined.\n");
      return;
   }
   
   if (fltr == NULL || 	       	/** need at least timefilter from 1st to last event used    **/
       fltr->t == NULL) {	/** to keep the, full observation period spanning, attitude **/
      /** data in check					    **/
      char ts[256];
      
      sprintf (ts, "%.10f-%.10f", exp->tstart, exp->tend);
      fltr= filter_time (fltr, ts);
   }
   else {
      FL_DRANGEREC ptl;
      
      ptl.mode= 1;
      ptl.r.s= exp->tstart;  ptl.r.L= exp->tend-exp->tstart; ptl.next= NULL;
      fltr->t = filter_tAnd (NULL, fltr->t, &ptl);
   }
   
   exp->sbp= cdi_initstab (exp, ins, 127, rsln);
   if ((afp= att_open (name, exp, fltr)) != NULL) {
      
      exp->attfile= strdup (afp->f->myName);
      
      Trace= (gbl_debug&DBG_VISIBMASK) >= DBG_L4;	/** Produce output, tracing the timewindows and	**/
      if (Trace) cdi_dbgout_hdr (DBG_L4);
      
      
      while ((att = (ATT_CAMERAPTR) afp->eRead (NULL, afp)) != NULL) {
	 
	 /*   if (afp->state&(EVTFLG_ATTINW|EVTFLG_ATTTWEY)) entering or within a time window .	**
	    **  											**
	    **  eRead only returns data falling within a time window, here we do not care		**
	    **  about transitions into or out of a window. AND, if there is no timewindow, meaning	**
	    **  DO read all events, the EVTFLG_ATTINW or EVTFLG_ATTTWEY are NEVER raised and all data **
	    **  was skipped. Now this no-timewindow situation occurs if the att. span is fully within **
	    **  1 tw, the evt_open then deletes the tw (no checks in the read).			**
	    */
	 
	 if (exp->sbp == NULL)
	    exp->sbp = cdi_initstab (exp, ins, 127, rsln);
	 
	 if (!(afp->state&EVTFLG_ATTTWEX)) {
	    if (exp->sbp->n == 0) 
	       exp->sbp->mjds= att->mjd;
	    exp->sbp->mjde= att->mjd;
	    
	    cdi_addstab (exp->sbp, att, ins, exp->wfc);
	    
	    if ( afp->tw_fil != NULL &&
		(att->mjd < afp->tw_fil->r.s || 
		 att->mjd > afp->tw_fil->r.s+afp->tw_fil->r.L)) {
	       
	       if (oowc == 0) {
		  sprintf (mess, "cdi_stability: out-of-timewindow ev. %.6f  [ %.6f - %.6f ]\n",
			   att->mjd, afp->tw_fil->r.s, afp->tw_fil->r.s+afp->tw_fil->r.L); 
		  wLog (DBG_WARNING, mess);
	       }
	       oowc++;
	    }
	 }
	 
	 if (Trace) {
	    if (afp->state&EVTFLG_ATTTWEX)	/** exiting a time window .......... **/
	       cdi_stab_dbgout (att->mjd, exp->tstart, NULL, "tw exit", 0.0);
	    if (afp->state&EVTFLG_ATTTWEY)	/** entering a time window **/
	       cdi_stab_dbgout (att->mjd, exp->tstart, NULL, "tw entry", 0.0);
	 }
      }
      
      if (oowc != 0) {
	 sprintf (mess,
		  "cdi_stability: a grand total %d out-of-any-timewindow events were processed!!\n",
		  oowc);
	 wLog (DBG_WARNING, mess);
      }
      cdi_finistab (exp->sbp, ins);
      exp->drift_mn= exp->sbp->mean;
      exp->drift_sg= exp->sbp->sigma; 
      exp->ndrift  = exp->sbp->n;
      
   }
   cdi_align (exp->sbp, ins);
   
   if (gbl_debug&DBG_DEVELOP) 
      cdi_alignreport (exp->sbp, ins);
}
		    
